-- Funktion sichert ein Element oder alle Elemente eines Typs aus Grafana in die Datenbank
CREATE OR REPLACE FUNCTION x_10_interfaces.grafana__elements__save_to_db(
    element_type varchar DEFAULT null,
    element_uid  varchar DEFAULT null,
    grafana_url  varchar DEFAULT tsystem.settings__get('Grafana_API_URL'),
    api_key      varchar DEFAULT tsystem.settings__get('Grafana_API_Token')
  )
  RETURNS VOID
  AS $$
  DECLARE
    api_response        jsonb;
    element             jsonb;
    element_data        jsonb;
    endpoints CONSTANT  jsonb := jsonb_build_object(
      'dashboard', jsonb_build_object(
          'list', '/api/search?type=dash-db',
          'detail', '/api/dashboards/uid/%s'
      ),
      'datasource', jsonb_build_object(
          'list', '/api/datasources',
          'detail', '/api/datasources/uid/%s'
      ),
      'folder', jsonb_build_object(
          'list', '/api/folders',
          'detail', '/api/folders/%s'
      --),
      --'alert_channel', jsonb_build_object(
      --    'list', '/api/alert-notifications',
      --    'detail', '/api/alert-notifications/%s'
      )
    );
    endpoint_list     text;
    endpoint_detail   text;
    element_type_key  text;
  BEGIN

    -- Wenn element_type NULL ist, alle unterstützten Elementtypen durchlaufen
    IF element_type IS NULL THEN
      -- Iteration über alle Elementtypen
      FOR element_type_key IN SELECT * FROM jsonb_object_keys(endpoints)
      LOOP
        -- Debug-Infos
        RAISE NOTICE 'x_10_interfaces.grafana__elements__save_to_db | element_type: % ', element_type_key;

        -- Endpunkte abrufen
        endpoint_list   := ( endpoints -> element_type_key ->> 'list' )::text;
        endpoint_detail := ( endpoints -> element_type_key ->> 'detail' )::text;

        -- Alle Elemente des aktuellen Typs abrufen und speichern
        PERFORM x_10_interfaces.grafana__elements__save_to_db(
            element_type    => element_type_key,
            grafana_url     => grafana_url,
            api_key         => api_key
        );
      END LOOP;
    ELSE
      -- Prüfen, ob der Element-Typ unterstützt wird
      IF NOT ( element_type IN (SELECT * FROM jsonb_object_keys(endpoints))) THEN
        RAISE EXCEPTION 'Unsupported element type: %', element_type;
      END IF;

      -- Endpunkte für den angegebenen Elementtyp abrufen
      endpoint_list   := ( endpoints -> element_type ->> 'list' )::text;
      endpoint_detail := ( endpoints -> element_type ->> 'detail' )::text;

      -- Wenn eine UID angegeben wurde, speichere nur dieses Element
      IF element_uid IS NOT NULL THEN
        PERFORM x_10_interfaces.grafana__element__save_to_db(
          grafana_url     => grafana_url,
          element_type    => element_type,
          element_uid     => element_uid,
          endpoint_detail => endpoint_detail,
          api_key         => api_key
        );
      ELSE
        -- Alle Elemente des angegebenen Typs abrufen und speichern
        SELECT content::jsonb
          INTO api_response
          FROM tsystem.http_request( (
                  'GET',
                  format('%s%s', grafana_url, endpoint_list),
                  ARRAY[http_header('Authorization', format('Bearer %s', api_key))],
                  null,
                  null
                )::http_request );

        -- Debug-Infos
        RAISE NOTICE 'x_10_interfaces.grafana__elements__save_to_db | api_response: % ', api_response;

        -- Iteration über die Elemente in der Liste
        FOR element IN SELECT jsonb_array_elements(api_response)
        LOOP
          -- UID aus dem Element extrahieren
          element_uid := coalesce(element ->> 'uid', element ->> 'id', null);

          -- Element speichern
          PERFORM x_10_interfaces.grafana__element__save_to_db(
            element_type    => element_type,
            element_uid     => element_uid,
            endpoint_detail => endpoint_detail,
            grafana_url     => grafana_url,
            api_key         => api_key
          );
        END LOOP;
      END IF;
    END IF;

END $$ LANGUAGE plpgsql;


-- Funktion speichert ein einzelnes Element aus Grafana in die Datenbank
CREATE OR REPLACE FUNCTION x_10_interfaces.grafana__element__save_to_db(
    element_type    varchar,
    element_uid     varchar,
    endpoint_detail varchar,
    grafana_url     varchar DEFAULT tsystem.settings__get('Grafana_API_URL'),
    api_key         varchar DEFAULT tsystem.settings__get('Grafana_API_Token')
  )
  RETURNS VOID
  AS $$
  DECLARE
    api_endpoint  varchar;
    api_response  jsonb;
    element_title varchar;
    element_kunde varchar;
  BEGIN

    -- Debug-Infos
    RAISE NOTICE 'x_10_interfaces.grafana__element__save_to_db | element_type: % | element_uid: %', element_type, element_uid;

    -- API-Endpunkt für das Detail-Element formatieren
    api_endpoint := format( '%s%s', grafana_url, format( endpoint_detail, element_uid ) );

    -- Detaildaten abrufen
    SELECT content::jsonb
      INTO api_response
      FROM tsystem.http_request( (
            'GET',
            api_endpoint,
            ARRAY[ http_header( 'Authorization', format( 'Bearer %s', api_key ) ) ],
            null,
            null
           )::http_request );

    -- Element-Titel extrahieren
    element_title := coalesce( api_response ->> 'title', api_response -> 'dashboard' ->> 'title', element_uid );
    element_kunde := coalesce( api_response ->> 'tags', null );

    -- Element in der Datenbank speichern
    INSERT INTO x_10_interfaces.grafana_elements (
      graf_elem_type,
      graf_elem_uid,
      graf_elem_title,
      graf_elem_kunde,
      graf_elem_json
    )
    VALUES (
      element_type,
      element_uid,
      element_title,
      element_kunde,
      api_response
    )
    ON CONFLICT (graf_elem_type, graf_elem_uid)
    DO UPDATE SET
      graf_elem_title = EXCLUDED.graf_elem_title,
      graf_elem_json  = EXCLUDED.graf_elem_json;
END $$ LANGUAGE plpgsql;


-- Funktion zum Importieren von Elementen in Grafana
CREATE OR REPLACE FUNCTION x_10_interfaces.grafana__elements__send_to_grafana(
    element_type  varchar DEFAULT null,  -- Optional: Um nur einen bestimmten Elementtyp zu importieren
    element_uid   varchar DEFAULT null,  -- Optional: Um nur ein spezifisches Element anhand der UID zu importieren
    grafana_url   varchar DEFAULT tsystem.settings__get('Grafana_API_URL'),
    api_key       varchar DEFAULT tsystem.settings__get('Grafana_API_Token')
  )
  RETURNS VOID
  AS $$
  DECLARE
    element             record;
    api_endpoint        varchar;
    api_response        jsonb;
    endpoints CONSTANT  jsonb := jsonb_build_object(
        'dashboard', '/api/dashboards/db',
        'datasource', '/api/datasources',
        'folder', '/api/folders',
        'alert_channel', '/api/alert-notifications'
    );
  BEGIN
    -- Filter für die zu importierenden Elemente basierend auf den übergebenen Parametern
    FOR element IN
      SELECT graf_elem_type, graf_elem_uid, graf_elem_json
        FROM x_10_interfaces.grafana_elements
       WHERE ( graf_elem_type = element_type OR element_type IS null )
         AND ( graf_elem_uid = element_uid OR element_uid IS null )
    LOOP
      -- Bestimmen des API-Endpunkts
      api_endpoint := endpoints ->> element.graf_elem_type;

      -- Sicherstellen, dass der Elementtyp unterstützt wird
      IF api_endpoint IS null THEN
        RAISE WARNING 'Unsupported element type: %, skipping import.', element.graf_elem_type;
        CONTINUE;
      END IF;

      -- Versuchen, das Element zu importieren
      PERFORM x_10_interfaces.grafana__element__send_to_grafana(
                api_endpoint  => api_endpoint,
                element_json  => element.graf_elem_json - 'meta' - 'id' #- '{dashboard,id}' || '{"overwrite": true}'::jsonb,
                grafana_url   => grafana_url,
                api_key       => api_key
              );

    END LOOP;

  EXCEPTION
    WHEN OTHERS THEN
      -- Zentrale Fehlerbehandlung
      RAISE WARNING 'Fehler beim Importieren von Elementen: %', SQLERRM;

  END $$ LANGUAGE plpgsql;


-- Hilfsfunktion zum Import eines einzelnen Elements
CREATE OR REPLACE FUNCTION x_10_interfaces.grafana__element__send_to_grafana(
    api_endpoint  varchar,
    element_json  jsonb,
    grafana_url   varchar DEFAULT tsystem.settings__get('Grafana_API_URL'),
    api_key       varchar DEFAULT tsystem.settings__get('Grafana_API_Token')
  )
  RETURNS VOID
  AS $$
  DECLARE
    api_response jsonb;
  BEGIN
    -- API-Aufruf per POST
    SELECT content::jsonb
      INTO api_response
      FROM tsystem.http_request( (
            'POST',
            format( '%s%s', grafana_url, api_endpoint ),
            ARRAY[
                http_header( 'Authorization', format( 'Bearer %s', api_key ) )
            ],
            'application/json',
            element_json::text
           )::http_request );

    -- Überprüfen des API-Antwortstatus
    IF api_response ->> 'status' IS DISTINCT FROM 'success' THEN
      RAISE WARNING 'Fehler beim Importieren des Elements, Status: % | element_json: %', api_response, element_json; --->> 'status';
    ELSE
      RAISE NOTICE 'Element erfolgreich importiert.';
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
      -- Fehler beim Import des einzelnen Elements
      RAISE WARNING 'Fehler beim Importieren des Elements: %', SQLERRM;

  END $$ LANGUAGE plpgsql;


-- API-Key für Grafana anlegen
CREATE OR REPLACE FUNCTION x_10_interfaces.grafana__service_account_api_key__create(
    grafana_username              varchar,
    grafana_password              varchar,
    grafana_service_account_name  varchar DEFAULT 'Prodat-ERP',
    grafana_url                   varchar DEFAULT tsystem.settings__get('Grafana_API_URL'),
    grafana_service_account_role  varchar DEFAULT 'Admin'  -- Grafana-Rolle: Viewer, Editor, Admin
  )
  RETURNS varchar
  AS $$
  DECLARE
    _auth_header         varchar;
    _sa_response         jsonb;
    _service_account_id  integer;
    _key_response        jsonb;
    _api_key             varchar;
  BEGIN

    -- Basic Auth für Grafana-API vorbereiten - Username:Passwort => Base64-codierter String
    _auth_header := 'Basic ' || encode(
                                  convert_to( grafana_username || ':' || grafana_password, 'UTF8' ),
                                  'base64'
                                );

    -- Service Account anlegen
    SELECT content::jsonb
      INTO _sa_response
      FROM tsystem.http_request( (
            'POST',
            grafana_url || '/api/serviceaccounts',
            ARRAY[ ( 'Authorization', _auth_header )::text ],
            'application/json',
            json_build_object(
              'name', grafana_service_account_name,
              'role', grafana_service_account_role,
              'basicAuthEnabled', true
            )::text
           )::http_request,
           1);

    -- ID des neu erstellten Service Accounts aus dem JSON holen
    _service_account_id := ( _sa_response ->>'id' )::int;
    IF _service_account_id IS NULL THEN
      RAISE EXCEPTION 'Fehler beim Anlegen des Service Accounts: %', _sa_response;
    END IF;

    -- API-Key für Service Account erstellen
    SELECT content::jsonb
      INTO _key_response
      FROM tsystem.http_request( (
            'POST',
            grafana_url || '/api/serviceaccounts/' || _service_account_id || '/tokens',
            ARRAY[ ( 'Authorization', _auth_header )::text ],
            'application/json',
            json_build_object(
              'name', 'API Key for ' || grafana_service_account_name
            )::text
           )::http_request,
           1);

    _api_key := _key_response ->> 'key';

    IF _api_key IS NULL THEN
      RAISE EXCEPTION 'Fehler beim Anlegen des Service Account API-Keys: %', _key_response;
    END IF;

    -- API-Key in Settings speichern
    PERFORM tsystem.settings__set('Grafana_API_Token', _api_key);

    -- API-Key zurückgeben, damit man ihn ggf. direkt weiterverwenden kann
    RETURN _api_key;

  END; $$ LANGUAGE plpgsql;